﻿using DicomObjects;
using DicomObjects.Enums;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Windows.Input;

namespace MauiSampleViewer;

// represents single node in our hierarchical tree (patient -> study -> series -> instance)
public class TreeNodeModel : INotifyPropertyChanged
{
    // visible label shown in ui for this node
    public string? DisplayName { get; set; }
    // list of identifiers to tract DICOM hierarchy
    public List<string?> Tags { get; set; } = new();
    // Depth of node (1 = Patient, 2 = Study)
    public int Level { get; set; }
    // Collection of child nodes (studies children of patient)
    public ObservableCollection<TreeNodeModel> Children { get; set; } = new();
    // tracks whether the node is expanded in ui
    private bool _isExpanded;
    public bool IsExpanded
    {
        get => _isExpanded;
        set { _isExpanded = value; OnPropertyChanged(nameof(IsExpanded)); }
    }

    // indicates whether the node is already exapanded
    private bool _canExpand;
    public bool CanExpand
    {
        get => _canExpand;
        set
        {
            if (_canExpand != value)
            {
                _canExpand = value;
                OnPropertyChanged(nameof(CanExpand));
            }
        }
    }

    // Ensures we load child data once per node
    public bool HasLoadedChildren { get; set; } = false;

    // Enables ui binding to auto updt visuals when properties like IsExpanded or CanExpand change.
    public event PropertyChangedEventHandler? PropertyChanged;
    protected void OnPropertyChanged(string name) =>
        PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(name));
}

public partial class RetrieveDialogType
{
    private DicomQuery? _query;
    private DicomDataSetCollection? _results;
    private readonly OptionsPage _options;
    private readonly DicomViewer _viewer;
    private readonly Window _window;

    private const string StudyText = "STUDY";
    private const string StudyListRetrievingMessage = "Please wait, while the study list is being retrieved";
    private const string SeriesRetrievingMessage = "Please wait, while the series list is being retrieved";
    private const string ImagesRetrievingMessage = "Please wait, while the Image list is retrieved";
    private const string InvalidLevel = "Invalid Level";
    private const string MoveStatusText = "Move Status -- ";
    public ObservableCollection<TreeNodeModel> TreeItems { get; set; } = new();
    public ICommand ToggleExpandCommand { get; }

    public RetrieveDialogType(OptionsPage options, DicomViewer viewer, Window window)
    {
        InitializeComponent();
        _options = options;
        _viewer = viewer;
        _window = window;

        TreeItems = new ObservableCollection<TreeNodeModel>();
        BindingContext = this; // makes properties from this calls accessible to XAML UI

        // command bound to expand/collapse gesture in the xaml icon
        ToggleExpandCommand = new Command<TreeNodeModel>(async (node) =>
        {
            if (!node.IsExpanded)
            {
                await ExpandNodeAsync(node);
                node.IsExpanded = true;
            }
            else
            {
                CollapseNode(node);
                node.IsExpanded = false;
            }
        });
    }

    private async void OnQueryClicked(object sender, EventArgs e)
    {
        try
        {
            _query = new DicomQuery()
            {
                Node = _options.QueryRemoteNode,
                Port = _options.QueryRemotePort,
                CallingAE = _options.QueryCallingAE,
                CalledAE = _options.QueryRemoteAE,
                Root = _options.UsePatientRoot ? QueryRoot.Patient : QueryRoot.Study,
                Level = _options.UsePatientRoot ? QueryLevel.PATIENT : QueryLevel.STUDY,
                Name = QueryPatientNameEntry.Text
            };

            _results = _query.Find();
            TreeItems.Clear();
            foreach (var ds in _results)
            {
                var label = _options.UsePatientRoot ? ds.Name : $"{ds.Name} / {GetDescription(ds)}";

                // Convert DICOM data to Tree nodess
                TreeItems.Add(new TreeNodeModel
                {
                    CanExpand = true,
                    DisplayName = label,
                    Level = _options.UsePatientRoot ? 1 : 2,
                    Tags =
                    [
                        _options.UsePatientRoot ? "1" : "2",
                        ds.PatientID,
                        ds.StudyUID
                    ],
                    Children = new ObservableCollection<TreeNodeModel>
                    {
                        new TreeNodeModel
                        {
                            DisplayName = _options.UsePatientRoot ? StudyListRetrievingMessage : SeriesRetrievingMessage
                        }
                    }
                });
            }

            ResultsView.ItemsSource = TreeItems;
        }
        catch (Exception ex)
        {
            await DisplayAlert("Query Error", ex.Message, "OK");
        }
    }

    private string GetDescription(DicomDataSet ds)
    {
        string s = ds.StudyDescription;
        if (string.IsNullOrEmpty(s))
        {
            s = StudyText;
        }

        if (ds[Keyword.StudyDate].ExistsWithValue)
        {
            s = $"{s} on {ds[Keyword.StudyDate].Value}";
        }
        return s;
    }

    private async void OnRetrieveClicked(object sender, EventArgs e)
    {
        DicomDataSetCollection dataSetCollection;
        DicomDataSet returnedStatus;

        if (ResultsView.SelectedItem is not TreeNodeModel selected)
            return;

        try
        {
            var tags = selected.Tags;
            if (_query != null)
            {
                _query.PatientID = tags.ElementAtOrDefault(1);
                _query.StudyUID = tags.ElementAtOrDefault(2);
                _query.SeriesUID = tags.ElementAtOrDefault(3);
                _query.InstanceUID = tags.ElementAtOrDefault(4);

                switch (selected.Level)
                {
                    case 1: // 1 = Patient
                        SetPatientValues(tags);
                        break;
                    case 2: // 2 = Study
                        SetStudyValues(tags);
                        break;
                    case 3: // 3 = Series
                        SetSeriesValues(tags);
                        break;
                    case 4: // 4 = Instance
                        SetInstanceValues(tags);
                        break;
                    default:
                        SetPatientValues(tags);
                        break;
                }

                if (_options.UseCGet)
                {
                    dataSetCollection = _query.Get();
                    for (int i = 0; i < dataSetCollection.Count; i++)
                    {
                        _viewer.Images.Add(new DicomImage(dataSetCollection[i]));
                    }
                }
                else
                {
                    _query.Destination = _options.QueryMoveDestination;
                    returnedStatus = _query.Move();
                    await DisplayAlert("Move Result", $"{MoveStatusText} {returnedStatus[0x0000, 0x0900].Value}", "OK");
                }
            }

            _viewer.AdjustMultiRowsColumns();
            Application.Current?.CloseWindow(_window);
        }
        catch (Exception ex)
        {
            await DisplayAlert("Retrieve Error", ex.Message, "OK");
        }
    }

    private void SetPatientValues(List<string?> tags)
    {
        if (_query != null)
        {
            _query.PatientID = tags.ElementAtOrDefault(1);
            _query.StudyUID = "";
            _query.SeriesUID = "";
            _query.InstanceUID = "";
            _query.Level = QueryLevel.PATIENT;
        }
    }

    private void SetStudyValues(List<string?> tags)
    {
        if (_query != null)
        {
            _query.PatientID = tags.ElementAtOrDefault(1);
            _query.StudyUID = tags.ElementAtOrDefault(2);
            _query.SeriesUID = "";
            _query.InstanceUID = "";
            _query.Level = QueryLevel.STUDY;
        }
    }

    private void SetSeriesValues(List<string?> tags)
    {
        if (_query != null)
        {
            _query.PatientID = tags.ElementAtOrDefault(1);
            _query.StudyUID = tags.ElementAtOrDefault(2);
            _query.SeriesUID = tags.ElementAtOrDefault(3);
            _query.InstanceUID = "";
            _query.Level = QueryLevel.SERIES;
        }
    }

    private void SetInstanceValues(List<string?> tags)
    {
        if (_query != null)
        {
            _query.PatientID = tags.ElementAtOrDefault(1);
            _query.StudyUID = tags.ElementAtOrDefault(2);
            _query.SeriesUID = tags.ElementAtOrDefault(3);
            _query.InstanceUID = tags.ElementAtOrDefault(4);
            _query.Level = QueryLevel.INSTANCE;
        }
    }

    private void OnResetClicked(object sender, EventArgs e)
    {
        ResultsView.ItemsSource = null;
    }

    private void OnCloseClicked(object sender, EventArgs e)
    {
        Application.Current?.CloseWindow(_window);
    }

    private Task ExpandNodeAsync(TreeNodeModel node)
    {
        if (node.HasLoadedChildren)
            return Task.CompletedTask;

        var tags = node.Tags;
        int tagLevel = node.Level;

        if (_query != null)
        {
            _query.PatientID = tags.ElementAtOrDefault(1);
            _query.StudyUID = tagLevel >= 2 ? tags.ElementAtOrDefault(2) : "";
            _query.SeriesUID = tagLevel >= 3 ? tags.ElementAtOrDefault(3) : "";
            _query.InstanceUID = "";

            switch (tagLevel)
            {
                case 1: _query.Level = QueryLevel.STUDY; break;
                case 2: _query.Level = QueryLevel.SERIES; break;
                case 3: _query.Level = QueryLevel.INSTANCE; break;
                default: return Task.CompletedTask;
            }

            var results = _query.Find();

            var children = new List<TreeNodeModel>();
            int nextLevel = tagLevel + 1;

            foreach (var ds in results)
            {
                var newTags = new List<string?> { nextLevel.ToString() };
                newTags.Add(tags.ElementAtOrDefault(1)); // PatientID

                newTags.Add(tagLevel == 1 ? ds.StudyUID : tags.ElementAtOrDefault(2));

                if (nextLevel >= 3)
                    newTags.Add(tagLevel == 2 ? ds.SeriesUID : tags.ElementAtOrDefault(3));

                if (nextLevel >= 4)
                    newTags.Add(ds.InstanceUID);

                string display = "Unknown";
                switch (nextLevel)
                {
                    case 2:
                        display = GetDescription(ds);
                        break;
                    case 3:
                        display = string.IsNullOrWhiteSpace(ds.SeriesDescription) ? "**SERIES**" : ds.SeriesDescription;
                        break;
                    case 4:
                        display = ds.InstanceUID;
                        break;
                }

                var child = new TreeNodeModel
                {
                    DisplayName = display,
                    Level = nextLevel,
                    Tags = newTags,
                    CanExpand = nextLevel < 4
                };

                if (child.CanExpand)
                    child.Children.Add(new TreeNodeModel { DisplayName = "Loading..." });

                children.Add(child);
            }

            node.HasLoadedChildren = true;
            int insertIndex = TreeItems.IndexOf(node);
            foreach (var child in children)
                TreeItems.Insert(++insertIndex, child);
        }

        var patientNode = FindPatientNode(node);
        ResultsView.ScrollTo(patientNode, position: ScrollToPosition.Start, animate: false);
        return Task.CompletedTask;
    }



    private void CollapseNode(TreeNodeModel node)
    {
        int index = TreeItems.IndexOf(node);
        int parentLevel = node.Level;

        index++;
        while (index < TreeItems.Count)
        {
            if (TreeItems[index].Level <= parentLevel)
                break;

            TreeItems.RemoveAt(index);
        }

        node.HasLoadedChildren = false;

        var patientNode = FindPatientNode(node);
        ResultsView.ScrollTo(patientNode, position: ScrollToPosition.Start, animate: false);
    }

    private TreeNodeModel FindPatientNode(TreeNodeModel node)
    {
        // Work backward from the current node until we find the patient (Level 1)
        int index = TreeItems.IndexOf(node);
        for (int i = index; i >= 0; i--)
        {
            if (TreeItems[i].Level == 1)
                return TreeItems[i];
        }
        return node; // fallback
    }
}

