﻿using DicomObjects;
using DicomObjects.UIDs;
using DicomObjects.Validation;

namespace MauiSampleViewer;
public partial class ExportImagePage : ContentPage
{
    private readonly DicomImage _image; 
    private readonly List<string> _availableFormats;

    public ExportImagePage(DicomImage selectedImage)
    {
        InitializeComponent();
        _image = selectedImage;

        _availableFormats = GetAllowedFormats();
        FormatPicker.ItemsSource = _availableFormats;
        FormatPicker.SelectedIndex = 0;
        FileNameEntry.Text = $"Export_{_image.InstanceUID}";
    }

    private List<string> GetAllowedFormats()
    {
        if (_image.SOPClass == SOPClasses.EncapsulatedPDF)
            return new() { ".pdf" };
        if (_image.FrameCount > 1)
            return new()
            {
                ".avi", ".mpeg", ".mpg", ".mp4", ".mov", ".gif"
            };
        return new()  
        {
            ".png", ".jpg", ".jpeg", ".webp"
        };
    }

    private async void OnExportClicked(object sender, EventArgs e)
    {
        var fileName = FileNameEntry.Text;
        var extension = FormatPicker.SelectedItem as string;

        if (string.IsNullOrWhiteSpace(fileName) || string.IsNullOrWhiteSpace(extension))
        {
            await DisplayAlert("Invalid Input", "Please enter a filename and select a format.", "OK");
            return;
        }

        string fullPath;
        string baseFileName = $"{fileName}{extension}";
            
        // this logic should probably be changed
        if (!string.IsNullOrWhiteSpace(CustomPathEntry.Text))
        {
            string input = CustomPathEntry.Text.Trim();

            if (_availableFormats.Any(ext => input.EndsWith(ext, StringComparison.OrdinalIgnoreCase)))
            {
                fullPath = input;
            }
            else
            {
                fullPath = Path.Combine(input, baseFileName);
            }
        }
        else
        {
            fullPath = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, baseFileName);
        }

        try
        {
            _image.Export(fullPath);
            await DisplayAlert(
                "Export Successful",
                $"File exported successfully to:\n{fullPath}",
                "OK"
            );
        }
        catch (Exception ex)
        {
            await DisplayAlert("Export Error", ex.Message, "OK");
        }
    }
}

